#!/bin/bash

#Validation Function:
#- Uses a regular expression to check if the domain name format is valid.

#Domain Pointing Check:
#- Uses the dig command to resolve the domain to an IP address.
#- Checks if the resolved IP address matches the server's IP address.

#Input and Validation:
#- Gets the server's IP address using the hostname -I command.
#- Prompts the user to enter a domain name.
#- Validates the domain name format.
#- Checks if the domain points to the server's IP address.
#- Displays appropriate messages based on the validation and check results.

# Function to validate the domain name
function validate_domain() {
    local domain="$1"
    
    # Check if the domain name format is valid
    if [[ "$domain" =~ ^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$ ]]; then
        return 0
    else
        echo "Invalid domain name format."
        return 1
    fi
}

# Function to check if the domain points to the server's IP
function check_domain_pointing() {
    local domain="$1"
    local server_ip="$2"
    
    # Get the IP address associated with the domain
    domain_ip=$(dig +short "$domain" | grep -m 1 '^[0-9]\{1,3\}\(\.[0-9]\{1,3\}\)\{3\}$')

    if [[ -z "$domain_ip" ]]; then
        echo "Domain does not resolve to any IP address."
        return 1
    fi

    if [[ "$domain_ip" == "$server_ip" ]]; then
        echo "Domain points to the server."
        return 0
    else
        echo "Domain does not point to the server."
        return 1
    fi
}

# Get server IP address
server_ip=$(hostname -I | awk '{print $1}')

# Get domain name input from the user
read -p "Enter domain name: " domain

# Validate the domain name
if validate_domain "$domain"; then
    # Check if the domain points to the server
    if check_domain_pointing "$domain" "$server_ip"; then
        echo "Domain is valid and points to the server."
    else
        echo "Domain is valid but does not point to the server."
    fi
else
    echo "Invalid domain name. Please try again."
fi
