#!/bin/bash

DB_FILE="/path/to/your/database.sqlite3"

# Function to initialize the database
function initialize_db() {
    sqlite3 "$DB_FILE" "CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT NOT NULL UNIQUE,
        password TEXT NOT NULL,
        domain TEXT NOT NULL,
        email TEXT NOT NULL
    );"
}

# Function to validate the email format
function validate_email_format() {
    local email="$1"
    if [[ "$email" =~ ^[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}$ ]]; then
        return 0
    else
        dialog --msgbox "Invalid email format." 6 40
        return 1
    fi
}

# Function to check if the email domain exists
function check_domain_exists() {
    local domain="$1"
    if dig +short "$domain" | grep -q '^[0-9]'; then
        return 0
    else
        dialog --msgbox "Domain does not exist." 6 40
        return 1
    fi
}

# Function to validate the domain name
function validate_domain() {
    local domain="$1"
    if [[ "$domain" =~ ^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$ ]]; then
        return 0
    else
        dialog --msgbox "Invalid domain name format." 6 40
        return 1
    fi
}

# Function to validate the username
function validate_username() {
    local username="$1"
    if [[ -z "$username" ]] || [[ ${#username} -lt 5 ]] || [[ ${#username} -gt 16 ]] || ! [[ "$username" =~ ^[a-zA-Z0-9]+$ ]] || id "$username" &>/dev/null; then
        dialog --msgbox "Invalid or existing username. Username must be 5-16 characters long, contain only letters and numbers, and cannot exist already." 6 40
        return 1
    fi
    return 0
}

# Function to validate the password
function validate_password() {
    local password="$1"
    local username="$2"
    if [[ -z "$password" ]] || [[ ${#password} -lt 5 ]] || [[ ${#password} -gt 16 ]] || [[ "$password" == "$username" ]] || [[ "$password" =~ \  ]] || ! [[ "$password" =~ [A-Z] ]] || ! [[ "$password" =~ [0-9] ]] || ! [[ "$password" =~ [^a-zA-Z0-9] ]]; then
        dialog --msgbox "Invalid password. Password must be 5-16 characters, not contain spaces, not be the same as the username, and must include at least one uppercase letter, one number, and one special character." 6 40
        return 1
    fi
    return 0
}

# Function to add user to SQLite database
function add_user_to_db() {
    local username="$1"
    local password="$2"
    local domain="$3"
    local email="$4"
    
    if sqlite3 "$DB_FILE" "SELECT 1 FROM users WHERE username='$username';" | grep -q 1; then
        dialog --msgbox "Username already exists!" 6 40
        return 1
    fi

    sqlite3 "$DB_FILE" "INSERT INTO users (username, password, domain, email) VALUES ('$username', '$password', '$domain', '$email');"
    dialog --msgbox "User added successfully!" 6 40
}

# Function to list users from SQLite database
function list_users() {
    sqlite3 "$DB_FILE" "SELECT * FROM users;" > /tmp/userlist
    dialog --textbox /tmp/userlist 22 76
}

# Function to delete user from SQLite database
function delete_user() {
    local username="$1"
    
    if sqlite3 "$DB_FILE" "DELETE FROM users WHERE username='$username';"; then
        dialog --msgbox "User deleted successfully!" 6 40
    else
        dialog --msgbox "Failed to delete user." 6 40
    fi
}

# Function to update user in SQLite database
function update_user() {
    local id="$1"
    local username="$2"
    local password="$3"
    local domain="$4"
    local email="$5"

    if sqlite3 "$DB_FILE" "UPDATE users SET username='$username', password='$password', domain='$domain', email='$email' WHERE id='$id';"; then
        dialog --msgbox "User updated successfully!" 6 40
    else
        dialog --msgbox "Failed to update user." 6 40
    fi
}

# Function to get user details for updating
function get_user_details() {
    local username="$1"
    
    sqlite3 "$DB_FILE" "SELECT * FROM users WHERE username='$username';" > /tmp/userdetails
    dialog --textbox /tmp/userdetails 22 76
}

initialize_db

while true; do
    dialog --title "User Management" --menu "Choose an action" 15 50 4 \
    1 "Add User" \
    2 "List Users" \
    3 "Update User" \
    4 "Delete User" \
    5 "Exit" 2> /tmp/menuitem

    choice=$(cat /tmp/menuitem)

    case $choice in
        1)
            dialog --title "Add User" --form "Enter details" 15 50 4 \
            "Username:" 1 1 "" 1 20 25 0 \
            "Password:" 2 1 "" 2 20 25 0 \
            "Domain:" 3 1 "" 3 20 25 0 \
            "Email:" 4 1 "" 4 20 25 0 2> /tmp/userform

            username=$(sed -n '1p' /tmp/userform)
            password=$(sed -n '2p' /tmp/userform)
            domain=$(sed -n '3p' /tmp/userform)
            email=$(sed -n '4p' /tmp/userform)

            if validate_username "$username" && validate_password "$password" "$username" && validate_domain "$domain" && validate_email_format "$email" && check_domain_exists "$(echo "$email" | awk -F'@' '{print $2}')"; then
                add_user_to_db "$username" "$password" "$domain" "$email"
            else
                dialog --msgbox "One or more fields are invalid. Please try again." 6 40
            fi
            ;;
        2)
            list_users
            ;;
        3)
            dialog --inputbox "Enter username of user to update:" 8 40 2> /tmp/username
            update_username=$(cat /tmp/username)
            get_user_details "$update_username"

            dialog --title "Update User" --form "Update details" 15 50 4 \
            "ID:" 1 1 "" 1 20 25 0 \
            "Username:" 2 1 "" 2 20 25 0 \
            "Password:" 3 1 "" 3 20 25 0 \
            "Domain:" 4 1 "" 4 20 25 0 \
            "Email:" 5 1 "" 5 20 25 0 2> /tmp/userform

            id=$(sed -n '1p' /tmp/userform)
            username=$(sed -n '2p' /tmp/userform)
            password=$(sed -n '3p' /tmp/userform)
            domain=$(sed -n '4p' /tmp/userform)
            email=$(sed -n '5p' /tmp/userform)

            if validate_username "$username" && validate_password "$password" "$username" && validate_domain "$domain" && validate_email_format "$email" && check_domain_exists "$(echo "$email" | awk -F'@' '{print $2}')"; then
                update_user "$id" "$username" "$password" "$domain" "$email"
            else
                dialog --msgbox "One or more fields are invalid. Please try again." 6 40
            fi
            ;;
        4)
            dialog --inputbox "Enter username of user to delete:" 8 40 2> /tmp/username
            delete_username=$(cat /tmp/username)
            delete_user "$delete_username"
            ;;
        5)
            break
            ;;
    esac
done

rm -f /tmp/menuitem /tmp/userform /tmp/username /tmp/userlist /tmp/userdetails
